package org.msh.tb.bd.tbforms.dhis2;

import org.jboss.seam.Component;
import org.msh.tb.application.TransactionManager;
import org.msh.tb.bd.Quarter;
import org.msh.tb.bd.tbforms.query.TbFormQuery;
import org.msh.tb.bd.tbforms.dhis2.data.DataValuePayload;
import org.msh.tb.entities.Tbunit;

import javax.persistence.EntityManager;
import java.util.List;

/**
 * Created by Mauricio on 17/02/2017.
 * Abstract class that should be inherited to create components
 * that will query the database and then mount the query result on a payload to send to DHIS2
 */
public abstract class TBFormBlockBuilder {

    protected TransactionManager transaction;

    /**
     * Builds the block controlling the transaction.
     * It will get the query result of the specialized queryDB method an then will mount this result
     * on a DTO DataValuePayload, using the specialized method mountResultPayload.
     * @param tbunit
     * @param selectedQuarter
     * @param payload
     */
    public void buildBlock(Tbunit tbunit, Quarter selectedQuarter, DataValuePayload payload){
        if (!getTransaction().isActive()) {
            getTransaction().begin();
        }

        // set all fields as zero, in case the information is being re send, to guarantee
        // that previous fields will be set to zero before setting the new value
        loadAllFieldsAsZero(payload);

        EntityManager em = getTransaction().getEntityManager();

        List<Object[]> queryResult = getBlockQuery().queryDB(tbunit, null, null, selectedQuarter, em);

        for (Object[] result : queryResult) {
            mountResultPayload(result, payload);
        }

        getTransaction().rollback();
    }

    /**
     * Builds the block controlling the transaction.
     * It will mount the payload based on the mockResults passed as parameters.
     * Should be used to test the link (DHIS2 ids) between eTBManager and DHIS2.
     * Should be used with the results generated by MockDataResources.
     * @param mockResults
     * @param payload
     */
    public void buildBlock(List<Object[]> mockResults, DataValuePayload payload){
        for (Object[] result : mockResults) {
            mountResultPayload(result, payload);
        }
    }

    /**
     * Initializes the payload setting all fields as zero.
     */
    protected abstract void loadAllFieldsAsZero(DataValuePayload payload);

    protected abstract TbFormQuery getBlockQuery();

    /**
     * Mount the query result into a DTO DataValuePayload, to be sent to DHIS2.
     * @param result
     * @param payload
     */
    protected abstract void mountResultPayload(Object[] result, DataValuePayload payload);

    /**
     * Start a new transaction
     */
    private void beginTransaction() {
        getTransaction().begin();
    }

    /**
     * Start a new transaction
     */
    private void rollbackTransaction() {
        getTransaction().rollback();
    }

    /**
     * Return the transaction in use by the task
     * @return
     */
    private TransactionManager getTransaction() {
        if (transaction == null)
            transaction = (TransactionManager) Component.getInstance("transactionManager");
        return transaction;
    }
}
